<?php
namespace Drupal\shareholder_register\Plugin\ShareholderRegisterExportPlugin;

use Drupal\Core\Link;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

use Drupal\shareholder_register\ShareholderSharesLine;
use Drupal\shareholder_register\ShareholderRegisterService;
use Drupal\shareholder_register\Plugin\ShareholderRegisterExportPluginInterface;

use PhpOffice\PhpSpreadsheet\Cell\DataType;

/**
 * Export Transactions and Conversions.
 *
 * @ShareholderRegisterExportPlugin(
 *   id = "transactions_conversions",
 *   label = @Translation("Transactions and Conversions"),
 *  )
 */
class TransactionsAndConversions extends PluginBase implements PluginFormInterface, ShareholderRegisterExportPluginInterface {
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {

    $form['date'] = [
      '#type' => 'date',
      '#title' => $this->t('End Date'),
      '#required' => TRUE,
      '#weight' => '0',
    ];
    $form['use_payment_date'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use date of payment'),
      '#default_value' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['form_values'] = $form_state->getValues();
  }

  /**
   * {@inheritdoc}
   */
  public function getExportBatch() {
    $config = [
      'date' => $this->configuration['form_values']['date'],
      'use_payment_date' => $this->configuration['form_values']['use_payment_date'],
      'compute_ids_callback' => 'Drupal\shareholder_register\ShareholderRegisterExportService::allValidShareholders',
      'compute_callback' => 'Drupal\shareholder_register\Plugin\ShareholderRegisterExportPlugin\TransactionsAndConversions::shareholderTransactionsConversions',
      'write_callback' => 'Drupal\shareholder_register\Plugin\ShareholderRegisterExportPlugin\TransactionsAndConversions::writeShareholderSharesLine',
      'page_title' => $this->t(
        'Transactions and conversions up to @date (@type)',
        [
          '@date' => $this->configuration['form_values']['date'],
          '@type' => $this->configuration['form_values']['use_payment_date'] ? $this->t('Date of payment') : $this->t('Date of validation'),
        ]),
      'column_headers' => [
        t('Date'),
        t('Shareholder Number'),
        t('Shareholder Name'),
        t('Quantity'),
        t('Share Numbers'),
      ],
    ];

    foreach (ShareholderRegisterService::getShareHashFields() as $field_data) {
      $config['column_headers'][] = $field_data['label'];
    }

    $this->shareholderRegisterExport = \Drupal::service('shareholder_register.export');
    $batch = $this->shareholderRegisterExport->getExportBatch($config);

    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public static function shareholderTransactionsConversions($config, $shareholder, &$context) {
    $formatter = \Drupal::service('shareholder_register.formatter');
    $service = \Drupal::service('shareholder_register.default');
    $query = \Drupal::service('shareholder_register.query');

    $lines = [];

    $transactions = $shareholder->getValidTransactionsInRange(NULL, $config['date'], $config['use_payment_date']);
    foreach ($transactions as $transaction) {
      $shares_by_hash = [];
      foreach ($query->getShareRevisionsAtDate($transaction->getShareIds(), $transaction->getDate()) as $share) {
        $hash = $service->getShareHash($share);
        if (!isset($shares_by_hash[$hash])) {
          $shares_by_hash[$hash] = [];
        }
        $shares_by_hash[$hash][] = $share->id();
      }
      foreach ($shares_by_hash as $hash => $line_data) {
        $context['results']['totals'][] = new ShareholderSharesLine(
          $transaction->getTransactionDate($config['use_payment_date']),
          $shareholder->id(),
          count($line_data) * ($transaction->getQuantity() > 0 ? 1 : -1),
          $line_data,
          $hash
        );
      }
    }

    $conversions = $shareholder->getConversions();
    foreach ($conversions as $conversion) {
      $change_parts = explode('---', $conversion['conversion_hash']);
      $old = $change_parts[0];
      $new = $change_parts[1];

      $context['results']['totals'][] = new ShareholderSharesLine(
        $conversion['date'],
        $shareholder->id(),
        count($conversion['share_ids']) * -1,
        $conversion['share_ids'],
        $old
      );
      $context['results']['totals'][] = new ShareholderSharesLine(
        $conversion['date'],
        $shareholder->id(),
        count($conversion['share_ids']),
        $conversion['share_ids'],
        $new
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function writeShareholderSharesLine($config, $line, $sheet, $row, &$context) {
    $service = \Drupal::service('shareholder_register.default');
    $formatter = \Drupal::service('shareholder_register.formatter');

    $shareholder = $line->getShareholder();

    $sheet->setCellValue("A{$row}", $line->getDate());
    $sheet->setCellValueExplicit("B{$row}", $shareholder->getNumber(), DataType::TYPE_STRING);
    $sheet->setCellValue("C{$row}", $shareholder->getName());
    $sheet->setCellValue("D{$row}", $line->getQuantity());
    $sheet->setCellValue("E{$row}", $formatter->shareIdsToRanges($line->getShareIds()));

    $col = 'F';
    foreach (ShareholderRegisterService::getShareHashTextComponents($line->getShareHash()) as $component) {
      $sheet->setCellValue("{$col}{$row}", $component);
      $col = chr(ord($col) + 1);
    }
  }

}
