<?php

namespace Drupal\shareholder_register;

use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Symfony\Component\Routing\Route;

/**
 * Provides routes for Share transaction entities.
 *
 * @see \Drupal\Core\Entity\Routing\AdminHtmlRouteProvider
 * @see \Drupal\Core\Entity\Routing\DefaultHtmlRouteProvider
 */
class ShareTransactionHtmlRouteProvider extends AdminHtmlRouteProvider {

  /**
   * {@inheritdoc}
   */
  public function getRoutes(EntityTypeInterface $entity_type) {
    $collection = parent::getRoutes($entity_type);

    $entity_type_id = $entity_type->id();

    if ($settings_form_route = $this->getSettingsFormRoute($entity_type)) {
      $collection->add("$entity_type_id.settings", $settings_form_route);
    }

    if ($report_route = $this->getReportRoute($entity_type)) {
      $collection->add("entity.$entity_type_id.extract", $report_route);
    }

    if ($request_route = $this->getRequestRoute($entity_type)) {
      $collection->add("entity.$entity_type_id.request", $request_route);
    }

    return $collection;
  }

  /**
   * Gets the settings form route.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getSettingsFormRoute(EntityTypeInterface $entity_type) {
    if (!$entity_type->getBundleEntityType()) {
      $route = new Route("/admin/structure/{$entity_type->id()}/settings");
      $route->setDefaults([
          '_form' => 'Drupal\shareholder_register\Form\ShareTransactionSettingsForm',
          '_title' => "{$entity_type->getLabel()} settings",
      ])
        ->setRequirement('_permission', $entity_type->getAdminPermission())
        ->setOption('_admin_route', TRUE);

      return $route;
    }
  }

  /**
   * Gets the register transaction route.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getRegisterTransactionRoute(EntityTypeInterface $entity_type) {
      $route = new Route($entity_type->getLinkTemplate('report'));
      $route->setDefaults([
          '_title' => "{$entity_type->getLabel()} ",
          '_controller' => '\Drupal\shareholder_register\Controller\ShareholderController::register_transaction',
      ])
        ->setRequirement('_permission', 'access shareholder revisions')
        ->setOption('_admin_route', TRUE);

      return $route;
  }

  /**
   * Gets the report route.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getReportRoute(EntityTypeInterface $entity_type) {
    $route = new Route($entity_type->getLinkTemplate('extract'));
    $route->setDefaults([
        '_title' => "{$entity_type->getLabel()} report",
        '_controller' => '\Drupal\shareholder_register\Controller\ShareTransactionController::downloadExtract',
    ])
      ->setRequirement('_permission', 'view all share transaction entities')
      ->setOption('_admin_route', FALSE);

    return $route;
  }

  /**
   * Gets the report route.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getRequestRoute(EntityTypeInterface $entity_type) {
    $route = new Route($entity_type->getLinkTemplate('request'));
    $route->setDefaults([
        '_title' => "{$entity_type->getLabel()} request",
        '_controller' => '\Drupal\shareholder_register\Controller\ShareTransactionController::downloadRequest',
    ])
      ->setRequirement('_permission', 'view all share transaction entities')
      ->setOption('_admin_route', FALSE);

    return $route;
  }

}
