<?php

namespace Drupal\shareholder_register;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Defines a class to build a listing of Share transaction entities.
 *
 * @ingroup shareholder_register
 */
class ShareTransactionListBuilder extends EntityListBuilder {


  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['name'] = $this->t('Transaction Number');
    $header['date'] = $this->t('Transaction Date');
    $header['payment_date'] = $this->t('Date of Payment');
    $header['shareholder_number'] = $this->t('Shareholder Number');
    $header['name'] = $this->t('Name');
    $header['quantity'] = $this->t('Quantity');
    $header['issue'] = $this->t('Share Issue');
    $header['shares'] = $this->t('Shares');
    $header['state'] = $this->t('State');

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /* @var $entity \Drupal\shareholder_register\Entity\ShareTransaction */

    $shareholder = $entity->getShareholder();
    $issue = $entity->getShareIssue();

    $row['name'] = Link::createFromRoute(
      $entity->label(),
      'entity.share_transaction.canonical',
      ['share_transaction' => $entity->id()]
    );
    $row['date'] = $entity->getDate();
    $row['payment_date'] = $entity->getPaymentDate();
    $row['shareholder_number'] = $shareholder ? $shareholder->getNumber() : '';
    $row['name'] = $shareholder ? $shareholder->getName() : '';
    $row['quantity'] = $entity->getQuantity();
    $row['issue'] = $issue ? $issue->getName() : '';
    $row['shares'] = \Drupal::service('shareholder_register.formatter')->shareIdsToRanges(
      $entity->getShareIds());
    $row['state'] = $entity->getState();

    return $row + parent::buildRow($entity);
  }

  /**
   * Gets Share Transaction default operations.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity the operations are for.
   *
   * @return array
   *   The array structure is identical to the return value of
   *   self::getOperations().
   */
  protected function getDefaultOperations(EntityInterface $entity) {
    $operations = parent::getDefaultOperations($entity);

    $operations['view'] = array(
      'title' => $this
        ->t('View'),
      'weight' => 1,
      'url' => $entity
        ->urlInfo('canonical'),
    );

    if ($entity->get('state')->value == 'draft') {

      $operations['share-transaction-request'] = [
        'title' => $this->t('Request'),
        'weight' => 10,
        'url' => $entity->urlInfo('request'),
      ];

      if ($entity->access('update')) {

        $operations['wkf-validate'] = [
          'title' => $this->t('Validate'),
          'weight' => 20,
          'url' => $entity->urlInfo('validate'),
        ];

        $operations['wkf-refuse'] = [
          'title' => $this->t('Refuse'),
          'weight' => 30,
          'url' => $entity->urlInfo('refuse'),
        ];
      }
    }

    if ($entity->get('state')->value == 'valid') {
      $operations['share-transaction-register-extract'] = [
        'title' => $this->t('Transaction Extract'),
        'weight' => 40,
        'url' => $entity->urlInfo('extract'),
      ];
    }

    return $operations;
  }

}
