<?php

namespace Drupal\shareholder_register;

use Psr\Log\LoggerInterface;

use Drupal\shareholder_register\Entity\Share;
use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register\Entity\ShareTransaction;
use Drupal\shareholder_register\Entity\ShareTransactionGroupType;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Class ShareholderRegisterDrushService.
 */
class ShareholderRegisterDrushService implements ShareholderRegisterDrushServiceInterface {

  use StringTranslationTrait;

  /**
   * Constructs a new ShareholderRegisterDrushService object.
   */
  public function __construct(LoggerInterface $logger) {
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public function updateIdentifiers() {
    $query = \Drupal::entityQuery('shareholder')
      ->condition('state', 'valid');

    foreach (Shareholder::loadMultiple($query->execute()) as $shareholder) {
      $new_identifier = $shareholder->computeIdentifiers()['identifier'];
      if ($new_identifier !== $shareholder->getIdentifier()) {
        drush_print("new identifier for shareholder {$shareholder->id()}: '{$shareholder->getIdentifier()}' -> '{$new_identifier}'");

        $shareholder->setIdentifier($new_identifier);
        if (!$shareholder->validateIdentifier()) {
          drush_print("DUPLICATE IDENTIFIER !");
        }
        $shareholder->save();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function updateShareholder($id) {
    $s = Shareholder::load($id);
    if (!$s) {
      drush_print("Invalid shareholder id !");
    }
    $s->save();
  }

  /**
   * {@inheritdoc}
   */
  public function exportCertificates() {
    $s = \Drupal::service('mail_template.default');

    $path = "certificates";
    if (file_exists($path)) {
      drush_print("File '{$path}' exists!");
      return 1;
    }
    mkdir($path);
    $path2 = "certificates-sorted";
    if (file_exists($path2)) {
      drush_print("File '{$path2}' exists!");
      return 1;
    }
    mkdir($path2);

    $shareholder_ids = \Drupal::EntityQuery('shareholder')
                     ->condition('current', TRUE)
                     ->condition('mail', '%@%', 'NOT LIKE')
                     ->execute();

    foreach (Shareholder::loadMultiple($shareholder_ids) as $shareholder) {
      if ($shareholder->getState() !== 'valid') {
        continue;
      }

      $context = ['shareholder' => $shareholder];

      $file = $s->pdfForRenderArrays(
        [$s->renderArrayForTemplate('shareholder_extract', $context)],
        $s->pdfOptionsForTemplate('shareholder_extract', $context));
      copy($file, "{$path}/certificate-{$shareholder->getNumber()}.pdf");

      $postal_code = preg_replace("/[\/]/", "", mb_strtolower(trim($shareholder->get('address')->postal_code)));
      $street = preg_replace("/[\/]/", "", mb_strtolower(trim($shareholder->get('address')->address_line1)));
      $premise = preg_replace("/[\/]/", "", mb_strtolower(trim($shareholder->get('address')->premise)));
      $subpremise = preg_replace("/[\/]/", "", mb_strtolower(trim($shareholder->get('address')->subpremise)));

      copy($file, "{$path2}/certificate-{$postal_code}-{$street}-{$premise}-{$subpremise}-{$shareholder->getNumber()}.pdf");
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateTransaction($id, $skip) {
    $transaction = ShareTransaction::load($id);
    $context = [
      'skip-issuing' => $skip,
    ];
    $transaction->actionValidate($transaction->getPaymentDate(), $context);
  }

  /**
   * {@inheritdoc}
   */
  public function cancelValidatedTransaction($id) {
    $transaction = ShareTransaction::load($id);
    $shareholder = $transaction->getShareholder();
    if (!drush_confirm("Are you sure you want to cancel transaction {$transaction->getName()} ({$transaction->id()}) for shareholder '{$shareholder->getNumber()}' ?")) {
      return;
    }

    $group = $transaction->getTransactionGroup();
    foreach ($group->getTransactions() as $t) {
      $t->set('state', 'cancel');
      $t->set('name', '');
      $t->save();
    }
    $group->set('state', 'draft');
    $group->set('number', '');
    $group->save();

    if ($group->getShareGroupTypeBaseType() == ShareTransactionGroupType::BASE_ISSUE) {
      foreach ($transaction->getShares() as $share) {
        $share->set('name', '');
        $share->set('state', 'draft');
        $share->save();
      }
    }

    return "Canceled transaction {$id}!";
  }

  /**
   * {@inheritdoc}
   */
  public function issueSharesForTransactions() {
    $t_ids = \Drupal::EntityQuery('share_transaction')
           ->condition('state', 'valid')
           ->execute();

    $s_ids = \Drupal::EntityQuery('share')
           ->condition('state', 'draft')
           ->condition('share_transaction_ids', $t_ids, 'IN')
           ->sort('id', 'ASC')
           ->range(0, 5000)
           ->execute();

    if (!count($s_ids)) {
      drush_set_error('all shares are issued');
      drush_set_context('DRUSH_EXIT_CODE', 5);
    }

    $connection = \Drupal::database();
    $transaction = $connection->startTransaction();

    try {
      foreach (array_chunk($s_ids, 500) as $chunk) {
        foreach (Share::loadMultiple($chunk) as $share) {
          $share->actionIssue();
        }
      }
    }
    catch (Exception $e) {
      $transaction->rollBack();
      throw $e;
    }
  }

  /**
   * Check sum of all transactions equals to 'issued' shares.
   */
  protected function checkCurrentShares() {
    $result = [[
        'msg' => $this->t("Total number of issued share equals total quantity of all valid transactions."),
        'result' => TRUE,
    ]];
    $total_from_transactions = \Drupal::database()->query("select sum(quantity) as total from {share_transaction} where state='valid'")->fetchObject()->total;
    $total_from_shares = \Drupal::database()->query("select count(*) as total from {share} where state='issued'")->fetchObject()->total;
    if ($total_from_transactions != $total_from_shares) {
      $this->logger->error("Sum of quantity for valid share transactions ({$total_from_transactions}) != number of issued shares ({$total_from_shares})!");
      $result[0]['result'] = FALSE;
    }
    return $result;
  }

  /**
   * Check attachment of shares to transactions.
   */
  protected function checkAttachedShares() {
    $result = [[
        'msg' => $this->t("Attached shares to valid transactions"),
        'result' => TRUE,
    ]];

    $total = \Drupal::database()->query("select count(*) as total from {share_transaction} inner join {share__share_transaction_ids} on ({share_transaction}.id = {share__share_transaction_ids}.share_transaction_ids_target_id) where {share_transaction}.state = 'valid' group by {share_transaction}.id, {share_transaction}.quantity having abs({share_transaction}.quantity) != count(*)")->fetchObject();
    if ($total && $total > 0) {
      $this->logger->error("{$total} transactions with invalid number of shares attached!");
      $result[0]['result'] = FALSE;
    }
    return $result;
  }

  /**
   * Check transaction groups.
   */
  protected function checkTransactionGroups() {
    $result = [[
        'msg' => $this->t("Transaction groups check"),
        'result' => TRUE,
    ]];

    $total = \Drupal::database()->query("select count(*) as total from {share_transaction} where state='valid' and transaction_group is NULL")->fetchObject()->total;
    if ($total > 0) {
      $this->logger->error("{$total} valid transactions without transaction group!");
      $result[0]['result'] = FALSE;
    }

    $issue_type_ids = \Drupal::service('shareholder_register.default')->getShareGroupTypeIdsForBaseType('issue');

    $total_shares = \Drupal::database()->query("select count(*) as total from {share} where state in ('issued', 'redeemed')")->fetchObject()->total;
    $total_issued = \Drupal::database()->query(
      "select sum(quantity) as total from {share_transaction} inner join {share_transaction_group} on ({share_transaction}.transaction_group = {share_transaction_group}.id) where {share_transaction}.state = 'valid' and {share_transaction_group}.type in (:issue_types[])",
      [
        ':issue_types[]' => $issue_type_ids,
      ]
    )->fetchObject()->total;

    if ($total_shares != $total_issued) {
      $this->logger->error("Number issued shares {$total_shares} != number of shares from issue transaction groups ({$total_issued})!");
      $result[0]['result'] = FALSE;
    }
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function integrityCheck() {
    return array_merge(
      $this->checkCurrentShares(),
      $this->checkAttachedShares(),
      $this->checkTransactionGroups()
    );
  }
}
