<?php

namespace Drupal\shareholder_register;

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Reader\Xlsx;

use Drupal\Core\Url;
use Drupal\Core\Link;

use Drupal\shareholder_register\Entity\Shareholder;

/**
 * Class ShareholderRegisterExportService.
 */
class ShareholderRegisterExportService implements ShareholderRegisterExportServiceInterface {

  /**
   * Constructs a new ShareholderRegisterExportService object.
   */
  public function __construct() {

  }

  /**
   * {@inheritdoc}
   */
  public static function initBatch($config, &$context) {
    $context['results']['details'] = [];
    $context['results']['totals'] = [];
    $context['results']['config'] = $config;
  }

  /**
   * {@inheritdoc}
   */
  public static function computeBatch($config, &$context) {
    if (!isset($context['sandbox']['shareholder_ids'])) {

      $shareholder_ids = call_user_func_array($config['compute_ids_callback'], [$config]);
      $context['sandbox']['shareholder_ids'] = $shareholder_ids;
      $context['sandbox']['shareholder_ids_count'] = count($context['sandbox']['shareholder_ids']);
      $context['results']['totals'] = [];
      $context['results']['details'] = [];
    }

    for ($i = 0; $i < 100; $i++) {
      if ($shareholder_id = array_pop($context['sandbox']['shareholder_ids'])) {
        $shareholder = Shareholder::load($shareholder_id);
        call_user_func_array($config['compute_callback'], [$config, $shareholder, $context]);
      }
    }

    if ($context['sandbox']['shareholder_ids_count'] > 0) {
      $context['finished'] = ($context['sandbox']['shareholder_ids_count'] - count($context['sandbox']['shareholder_ids'])) / $context['sandbox']['shareholder_ids_count'];
    }
    else {
      $context['finished'] = 1;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function writeBatch($config, &$context) {
    if (!isset($context['results']['output'])) {
      $output = file_save_data('', 'private://export.xlsx');
      $output->setTemporary();
      $output->save();

      $context['results']['output'] = $output;

      $spreadsheet = new Spreadsheet();

      // Set document properties.
      $spreadsheet->getProperties()->setCreator('DSR')
        ->setLastModifiedBy(t('DSR'))
        ->setTitle(t('DSR Export'))
        ->setSubject(t('DSR Export'))
        ->setDescription(t('DSR Export'))
        ->setCategory(t('DSR'));
    }
    else {
      $reader = new Xlsx();
      $reader->setReadDataOnly(TRUE);
      $spreadsheet = $reader->load(drupal_realpath($context['results']['output']->getFileUri()));
    }

    if (!isset($context['sandbox']['to_write'])) {
      $config = $context['results']['config'];
      $sheet = $spreadsheet->setActiveSheetIndex(0);

      if (isset($config['page_title'])) {
        $sheet->setCellValue("A1", $config['page_title']);
      }

      if (isset($config['column_headers_callback'])) {
          call_user_func_array($config['column_headers_callback'], [$config, $sheet, $context]);
      }
      else {
        $col = 'A';
        foreach ($config['column_headers'] as $header) {
          $sheet->setCellValue("{$col}3", $header);
          $col = chr(ord($col) + 1);
        }
      }
      $context['sandbox']['row'] = 5;
      $context['sandbox']['to_write'] = $context['results']['totals'];
    }
    else {
      $sheet = $spreadsheet->getActiveSheet();
    }

    for ($i = 0; $i < 5000; $i++) {
      if ($group = array_pop($context['sandbox']['to_write'])) {
        $row = $context['sandbox']['row'];
        call_user_func_array($config['write_callback'], [$config, $group, $sheet, $row, $context]);
        $context['sandbox']['row']++;
      }
    }

    $writer = IOFactory::createWriter($spreadsheet, 'Xlsx');
    $writer->setPreCalculateFormulas(FALSE);
    $writer->save(drupal_realpath($context['results']['output']->getFileUri()));
    $context['results']['output']->save();

    if (count($context['results']['totals']) > 0) {
      $context['finished'] = (count($context['results']['totals']) - count($context['sandbox']['to_write'])) / count($context['results']['totals']);
    }
    else {
      $context['finished'] = 1;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function finishBatch($success, $results, $operations) {
    drupal_set_message(t(
        'Download export: @url',
        [
          '@url' => Link::fromTextAndUrl('export.xlsx', Url::fromUri(file_create_url($results['output']->getFileUri())))->toString(),
        ]
    ));
  }

  /**
   * {@inheritdoc}
   */
  public function getExportBatch($config) {
    $batch = array(
      'title' => t('Computing shareholder register state at date'),
      'operations' => [],
    );
    $batch['operations'][] = [
      'Drupal\shareholder_register\ShareholderRegisterExportService::initBatch',
      [$config],
    ];
    $batch['operations'][] = [
      'Drupal\shareholder_register\ShareholderRegisterExportService::computeBatch',
      [$config],
    ];
    $batch['operations'][] = [
      'Drupal\shareholder_register\ShareholderRegisterExportService::writeBatch',
      [$config],
    ];
    $batch['finished'] = 'Drupal\shareholder_register\ShareholderRegisterExportService::finishBatch';

    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public static function allValidShareholders($config) {
    return \Drupal::entityQuery('shareholder')
      ->condition('state', 'valid')
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public static function shareholderAtDate($config, $shareholder, &$context) {
    $formatter = \Drupal::service('shareholder_register.formatter');

    $share_ids = $shareholder->getShareIdsAtDate($config['date'], NULL, $config['use_payment_date']);
    if (!count($share_ids)) {
      return;
    }
    $context['results']['totals'][$shareholder->id()] = [
      'shareholder_id' => $shareholder->id(),
      'share_ids' => $share_ids,
      'shares' => $formatter->shareIdsToRanges($share_ids),
      'count' => count($share_ids),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function writeShareholderAtDate($config, $group, $sheet, $row, &$context) {
    $shareholder = Shareholder::load($group['shareholder_id']);

    $sheet->setCellValueExplicit("A{$row}", $shareholder->getNumber(), DataType::TYPE_STRING);
    $sheet->setCellValue("B{$row}", $shareholder->getName());
    $sheet->setCellValue("C{$row}", $shareholder->getMail());
    $sheet->setCellValue("D{$row}", $shareholder->getRegistrationDate());
    $sheet->setCellValue("E{$row}", $shareholder->getAddress()->address_line1);
    $sheet->setCellValueExplicit("F{$row}", $shareholder->getAddress()->premise, DataType::TYPE_STRING);
    $sheet->setCellValueExplicit("G{$row}", $shareholder->getAddress()->subpremise, DataType::TYPE_STRING);
    $sheet->setCellValueExplicit("H{$row}", $shareholder->getAddress()->postal_code, DataType::TYPE_STRING);
    $sheet->setCellValue("I{$row}", $shareholder->getAddress()->locality);
    $sheet->setCellValue("J{$row}", $shareholder->getAddress()->country_code);
    $sheet->setCellValueExplicit("K{$row}", $group['shares'], DataType::TYPE_STRING);
    $sheet->setCellValue("L{$row}", $group['count']);
  }
}
