<?php

namespace Drupal\shareholder_register;

use Drupal\Core\Datetime\DateFormatterInterface;

use Drupal\shareholder_register\Entity\ShareType;

/**
 * Class ShareholderRegisterFormatterService.
 */
class ShareholderRegisterFormatterService implements ShareholderRegisterFormatterServiceInterface {

  /**
   * Drupal\Core\Datetime\DateFormatterInterface definition.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs a new ShareholderRegisterFormatterService object.
   */
  public function __construct(
    DateFormatterInterface $dateFormatter) {

    $this->dateFormatter = $dateFormatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function numbersToRanges($numbers) {
    sort($numbers);
    $result = [];

    $start_range = $end_range = reset($numbers);
    while (($number = next($numbers)) !== FALSE) {
      if ($number == $end_range + 1) {
        $end_range = $number;
      }
      elseif ($number > $end_range + 1) {
        if ($start_range == $end_range) {
          $result[] = "{$start_range}";
        }
        else {
          $result[] = "{$start_range}-{$end_range}";
        }
        $start_range = $end_range = $number;
      }
    }
    if ($start_range !== FALSE && $start_range == $end_range) {
      $result[] = "{$start_range}";
    }
    else {
      $result[] = "{$start_range}-{$end_range}";
    }

    return implode(', ', $result);
  }

  /**
   * {@inheritdoc}
   */
  public function sharesToRanges($shares) {
    $numbers = array_map(
      function ($s) {
        return $s->getName();
      },
      $shares);
    return ShareholderRegisterFormatterService::numbersToRanges($numbers);
  }

  /**
   * {@inheritdoc}
   */
  public function shareIdsToRanges($share_ids) {
    if (!count($share_ids)) {
      return '';
    }
    $connection = \Drupal::database();
    $result = $connection->query("select name from {share} where id IN (:ids[])", [
                ':ids[]' => $share_ids,
    ]);
    $numbers = $result->fetchCol(0);
    return ShareholderRegisterFormatterService::numbersToRanges($numbers);
  }

  /**
   * Returns previous day (this is the incl. end date of a given period).
   */
  public function endDateIncl($date) {
    return date('Y-m-d', strtotime("{$date} previous day"));
  }

  /**
   * {@inheritdoc}
   */
  public function formatDate($date, $langcode = NULL) {
    if (empty($date)) {
      return '';
    }

    return $this->dateFormatter->format(
      is_numeric($date) ? $date : strtotime($date),
      'shareholder_register_date',
      '',
      NULL,
      $langcode
    );
  }

  /**
   * Rounds the given number.
   *
   * Replicates PHP's support for rounding to the nearest even/odd number
   * even if that number is decimal ($precision > 0).
   *
   * Copied from drupal commerce / calculator.
   *
   * @param string $number
   *   The number.
   * @param int $precision
   *   The number of decimals to round to.
   * @param int $mode
   *   The rounding mode. One of the following constants: PHP_ROUND_HALF_UP,
   *   PHP_ROUND_HALF_DOWN, PHP_ROUND_HALF_EVEN, PHP_ROUND_HALF_ODD.
   *
   * @return string
   *   The rounded number.
   *
   * @throws \InvalidArgumentException
   *   Thrown when an invalid (non-numeric or negative) precision is given.
   */
  public static function round(string $number, int $precision = 0, int $mode = PHP_ROUND_HALF_UP) : string {
    self::assertNumberFormat($number);
    if (!is_numeric($precision) || $precision < 0) {
      throw new \InvalidArgumentException('The provided precision should be a positive number');
    }

    // Round the number in both directions (up/down) before choosing one.
    $rounding_increment = bcdiv('1', pow(10, $precision), $precision);
    if (self::compare($number, '0') == 1) {
      $rounded_up = bcadd($number, $rounding_increment, $precision);
    }
    else {
      $rounded_up = bcsub($number, $rounding_increment, $precision);
    }
    $rounded_down = bcsub($number, 0, $precision);
    // The rounding direction is based on the first decimal after $precision.
    $number_parts = explode('.', $number);
    $decimals = !empty($number_parts[1]) ? $number_parts[1] : '0';
    $relevant_decimal = isset($decimals[$precision]) ? $decimals[$precision] : 0;
    if ($relevant_decimal < 5) {
      $number = $rounded_down;
    }
    elseif ($relevant_decimal == 5) {
      if ($mode == PHP_ROUND_HALF_UP) {
        $number = $rounded_up;
      }
      elseif ($mode == PHP_ROUND_HALF_DOWN) {
        $number = $rounded_down;
      }
      elseif ($mode == PHP_ROUND_HALF_EVEN) {
        $integer = bcmul($rounded_up, pow(10, $precision), 0);
        $number = bcmod($integer, '2') == 0 ? $rounded_up : $rounded_down;
      }
      elseif ($mode == PHP_ROUND_HALF_ODD) {
        $integer = bcmul($rounded_up, pow(10, $precision), 0);
        $number = bcmod($integer, '2') != 0 ? $rounded_up : $rounded_down;
      }
    }
    elseif ($relevant_decimal > 5) {
      $number = $rounded_up;
    }

    return $number;
  }

  /**
   * Compares the first number to the second number.
   *
   * @param string $first_number
   *   The first number.
   * @param string $second_number
   *   The second number.
   * @param int $scale
   *   The maximum number of digits after the decimal place.
   *   Any digit after $scale will be truncated.
   *
   * @return int
   *   0 if both numbers are equal, 1 if the first one is greater, -1 otherwise.
   */
  public static function compare(string $first_number, string $second_number, int $scale = 6) : int {
    self::assertNumberFormat($first_number);
    self::assertNumberFormat($second_number);
    return bccomp($first_number, $second_number, $scale);
  }

  /**
   * Assert that the given number is a numeric string value.
   *
   * @param string $number
   *   The number to check.
   *
   * @throws \InvalidArgumentException
   *   Thrown when the given number is not a numeric string value.
   */
  public static function assertNumberFormat($number) {
    if (is_float($number)) {
      throw new \InvalidArgumentException(sprintf('The provided value "%s" must be a string, not a float.', $number));
    }
    if (!is_numeric($number)) {
      throw new \InvalidArgumentException(sprintf('The provided value "%s" is not a numeric value.', $number));
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function conversionHashToText($change) {
    $change_parts = explode('---', $change);
    $old = $change_parts[0];
    $new = $change_parts[1];

    $old_components = ShareholderRegisterService::getShareHashTextComponents($old);
    $new_components = ShareholderRegisterService::getShareHashTextComponents($new);

    $changed_old_components = [];
    $changed_new_components = [];
    for ($i=0; $i<count($old_components); $i++) {
      if ($old_components[$i] !== $new_components[$i]) {
        $changed_old_components[] = $old_components[$i];
        $changed_new_components[] = $new_components[$i];
      }
    }
    $old_label = implode(' / ', $changed_old_components);
    $new_label = implode(' / ', $changed_new_components);

    return "{$old_label} -> {$new_label}";
  }
}
