<?php

namespace Drupal\shareholder_register;

use Drupal\shareholder_register\Entity\Shareholder;

/**
 * Class ShareholderRegisterQueryService.
 */
class ShareholderRegisterQueryService implements ShareholderRegisterQueryServiceInterface {

  /**
   * Constructs a new ShareholderRegisterQueryService object.
   */
  public function __construct() {

  }

  /**
   * Get the share ids held by $shareholder_id at date $date.
   *
   * @param int $shareholder_id
   *   The shareholder ID.
   * @param \DateTime $date
   *   The last included date.
   * @param $end_transaction
   *   The last included transaction.
   */
  public function getShareIdsAtDate($shareholder_id, $date = NULL, $end_transaction = NULL, $use_payment_date = FALSE) {
    $connection = \Drupal::database();

    $date_field = "date";
    if ($use_payment_date) {
      $date_field = "payment_date";
    }

    if ($date === NULL) {
      $result = $connection->query(
        "select max(coalesce({$date_field}, date)) as d from {share_transaction} where state = 'valid'")->fetchObject();
      $date = $result->d;
    }
    elseif ($date instanceof DateTime) {
      $date = $date->format('Y-m-d');
    }

    $result = $connection->query(
      <<<EOF
        select
          sts.entity_id as share_id
        from
          {share__share_transaction_ids} sts
          inner join {share_transaction} st on (sts.share_transaction_ids_target_id = st.id)
        where st.state = 'valid'
          and st.shareholder_id = :shareholder
          and coalesce(st.{$date_field}, st.date) <= :date
        group by sts.entity_id
        having sum(sign(st.quantity)) = 1
EOF
      , [
        ':shareholder' => $shareholder_id,
        ':date' => $date,
      ]);

    $r = $result->fetchCol();
    return $r;
  }

  /**
   * Get list of all dates on which conversions took place for a shareholder.
   */
  public function getConversionDatesForShareholder($shareholder_id) {
    $connection = \Drupal::database();

    // Get all shares ever held.
    $result = $connection->query(
      <<<EOF
        select
          distinct sts.entity_id as share_id
        from
          {share__share_transaction_ids} sts
          inner join {share_transaction} st on (sts.share_transaction_ids_target_id = st.id)
        where st.state = 'valid'
          and st.shareholder_id = :shareholder
          and st.quantity > 0
        group by sts.entity_id
EOF
      , [
        ':shareholder' => $shareholder_id,
      ]);

    $share_ids = $result->fetchCol();

    // Retrieve all conversion dates for these shares.

    $result = $connection->query(
      <<<EOF
        select
          distinct revision_date
        from
          {share_revision} rev
        where id in (:share_ids[])
        and revision_date is not NULL
EOF
      , [
        ':share_ids[]' => $share_ids,
      ]);

    $conversion_dates = $result->fetchCol();

    // Filter by periods that the shares are held.
    $filtered_conversion_dates = [];
    foreach ($conversion_dates as $conversion_date) {
      $result = $connection->query(
        <<<EOF
          select
            id
          from
            {share_revision} rev
          where revision_date = :revision_date
EOF
      , [
        ':revision_date' => $conversion_date,
      ]);
      $shares = $result->fetchCol();

      if (array_intersect($shares, $this->getShareIdsAtDate($shareholder_id, $conversion_date))) {
        $filtered_conversion_dates[] = $conversion_date;
      }
    }

    return $filtered_conversion_dates;
  }

  public function getRevisionAtDate($share_id, $date) {
    $connection = \Drupal::database();

    $result = $connection->query(
      <<<EOF
        select
          vid
        from
          {share_revision} rev
        where id = :share_id
        and IFNULL(revision_date, 0) <= :date
        order by IFNULL(revision_date, 0) desc
EOF
      , [
        ':share_id' => $share_id,
        ':date' => $date,
      ]);

    $revisions = $result->fetchCol();
    return reset($revisions);
  }

  /**
   * {@inheritdoc}
   */
  public function getShareRevisionsAtDate($share_ids, $date) {
    $storage = \Drupal::entityTypeManager()->getStorage('share');

    $shares = [];
    foreach ($share_ids as $share_id) {
      $shares[] = $storage->loadRevision(
        $this->getRevisionAtDate($share_id, $date));
    }
    return $shares;
  }

  /**
   * {@inheritdoc}
   */
  public function getConversionSummaryAtDate($shareholder_id, $date) {
    $connection = \Drupal::database();
    $storage = \Drupal::entityTypeManager()->getStorage('share');
    $summary = [];

    // Retrieve shares at date.
    $shareholder = Shareholder::load($shareholder_id);
    $share_ids = $shareholder->getShareIdsAtDate($date);

    $previous_day = date('Y-m-d', strtotime('-1 day', strtotime($date)));

    // Retrieve revisions at date.
    foreach ($share_ids as $share_id) {
      $new = $storage->loadRevision($this->getRevisionAtDate($share_id, $date));
      $old = $storage->loadRevision($this->getRevisionAtDate($share_id, $previous_day));

      $new_hash = ShareholderRegisterService::getShareHash($new);
      $old_hash = ShareholderRegisterService::getShareHash($old);
      if ($old_hash !== $new_hash) {
        $change_summary = "{$old_hash}---{$new_hash}";
        if (!isset($summary[$change_summary])) {
          $summary[$change_summary] = [];
        }

        $summary[$change_summary][] = $share_id;
      }
    }

    // Summarize changes.
    $summary_text = [];
    foreach ($summary as $change => $share_ids) {
      $summary_text[] = [
        'share_ids' => $share_ids,
        'share_numbers' => \Drupal::service('shareholder_register.formatter')->shareIdsToRanges($share_ids),
        'conversion_hash' => $change,
        'conversion_text' => \Drupal::service('shareholder_register.formatter')->conversionHashToText($change),
      ];
    }

    return $summary_text;
  }
}
