<?php

namespace Drupal\shareholder_register;

use Drupal\Core\Link;
use Drupal\Core\Url;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

use Drupal\shareholder_register\Entity\Share;
use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register\Entity\ShareTransaction;
use Drupal\shareholder_register\Entity\ShareIssue;
use Drupal\shareholder_register\Entity\ShareType;
use Drupal\shareholder_register\Entity\ShareTransactionGroupType;

/**
 * Class ShareholderRegisterService.
 */
class ShareholderRegisterService implements ShareholderRegisterServiceInterface {

  use StringTranslationTrait;

  /**
   * Drupal\Core\Messenger\MessengerInterface definition.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new ShareholderRegisterService object.
   */
  public function __construct(MessengerInterface $messenger) {
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public function checkShareholderRegisterConfig() {
    $result = [];

    $config = \Drupal::config('shareholder_register.settings');

    $config_url = Url::fromRoute('shareholder_register.shareholder_register_config_form');
    $link = Link::fromTextAndUrl($this->t('Configure it here'), $config_url);

    if (!trim($config->get('company_name'))) {
      $result[] = [
        'error', $this->t('Company name is not set! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }
    if (!trim($config->get('company_address'))) {
      $result[] = [
        'error', $this->t('Company address is not set! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }
    if (!trim($config->get('company_zip'))) {
      $result[] = [
        'error', $this->t('Company postal code is not set! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }
    if (!trim($config->get('company_locality'))) {
      $result[] = [
        'error', $this->t('Company locality is not set! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }
    if (!trim($config->get('company_id'))) {
      $result[] = [
        'error', $this->t('Company ID is not set! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }
    if (!trim($config->get('company_date_of_incorporation'))) {
      $result[] = [
        'error', $this->t('Date of Incorporation is not set! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function validateIdentifier(string $identifier, int $existing_shareholder_id = NULL) {
    $query = \Drupal::entityQuery('shareholder')
      ->condition('identifier', $identifier)
      ->condition('state', 'valid');
    if ($existing_shareholder_id) {
      $query = $query->condition('id', $existing_shareholder_id, '<>');
    }
    $values = $query->execute();
    return !(bool) count($values);
  }

  /**
   * {@inheritdoc}
   */
  public function sharesTotalValue(array $shares) {
    $total = '0';
    foreach ($shares as $share) {
      $total = bcadd($total, $share->getShareValue(), 2);
      $total = bcadd($total, $share->getIssuePremium(), 2);
    }
    return $total;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareGroupTypeIdsForBaseType(string $baseType) {
    if (!in_array($baseType, [
      ShareTransactionGroupType::BASE_ISSUE,
      ShareTransactionGroupType::BASE_TRANSFER,
      ShareTransactionGroupType::BASE_REDEMPTION,
    ])) {
      throw new ShareholderRegisterException("Invalid share group base type '{$baseType}'!");
    }

    return \Drupal::entityQuery('share_transaction_group_type')
      ->condition('baseType', $baseType)
      ->execute();
  }


  /**
   * {@inheritdoc}
   */
  public function getShareIdsAtDate($date, $end_transaction = NULL, $use_payment_date = FALSE) {
    $connection = \Drupal::database();

    $date_field = "date";
    if ($use_payment_date) {
      $date_field = "payment_date";
    }

    $issue_type_ids = $this->getShareGroupTypeIdsForBaseType(
      ShareTransactionGroupType::BASE_ISSUE);
    $redemption_type_ids = $this->getShareGroupTypeIdsForBaseType(
      ShareTransactionGroupType::BASE_REDEMPTION);

    $result = $connection->query(
      <<<EOF
        select
          sts.entity_id as share_id
        from
          {share__share_transaction_ids} sts
          inner join {share_transaction} st on (sts.share_transaction_ids_target_id = st.id)
          left join {share_transaction_group} stg on (st.transaction_group = stg.id)
        where st.state = 'valid'
          and st.{$date_field} <= :date
          and (stg.type in (:issue_types[])
               or (stg.type is NULL and st.quantity > 0))
        and sts.entity_id not in (
        select
          sts.entity_id as share_id
        from
          {share__share_transaction_ids} sts
          inner join {share_transaction} st on (sts.share_transaction_ids_target_id = st.id)
          inner join {share_transaction_group} stg on (st.transaction_group = stg.id)
        where st.state = 'valid'
          and st.{$date_field} <= :date
          and (stg.type in (:redemption_types[])
               or (stg.type is NULL and st.quantity < 0))
        )
EOF
      , [
        ':date' => $date,
        ':issue_types[]' => $issue_type_ids,
        ':redemption_types[]' => $redemption_type_ids,
      ]);

    return $result->fetchCol(0);
  }

  /**
   * {@inheritdoc}
   */
  public function getTotalShareValueAtDate($date, $end_transaction = NULL, $use_payment_date = FALSE) {
    $share_ids = $this->getShareIdsAtDate($date, $end_transaction, $use_payment_date);
    return $this->sharesTotalValue(Share::loadMultiple($share_ids));
  }

  /**
   * {@inheritdoc}
   */
  public function createIssueTransaction(Shareholder $shareholder, ShareIssue $share_issue, $quantity, $account, $payment_date = NULL) {
    $share_transaction = ShareTransaction::create(
      [
        'shareholder_id' => $shareholder->id(),
        'share_type_id' => $share_issue->getShareType()->id(),
        'share_issue_id' => $share_issue->id(),
        'quantity' => $quantity,
        'payment_date' => $payment_date ? $payment_date : date('Y-m-d'),
        'issue_premium' => $share_issue->getShareType()->getIssuePremium(),
        // Account to book transaction on.
        'field_account' => $account->id(),
    ]);
    $share_transaction->save();
    return $share_transaction;
  }

  /**
   * {@inheritdoc}
   */
  public static function getShareHashFields() {
    $fields = [[
        'name' => 'share_type_id',
        'label' => 'Type',
        'type' => 'entity_reference',
    ]];

    $hash_fields = \Drupal::config(
      'shareholder_register.settings')->get(
        'share_hash_fields');
    if ($hash_fields) {
      foreach ($hash_fields as $hash_field) {
        $fields[] = [
          'name' => $hash_field['field_name'],
          'label' => $hash_field['field_label'],
          'type' => $hash_field['field_type'],
        ];
      }
    }
    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public static function getShareHash($share) {
    $hash_components = [];
    foreach (static::getShareHashFields() as $hash_field) {
      if (!in_array($hash_field['type'], ['entity_reference', 'boolean'], TRUE)) {
        continue;
      }
      if ($hash_field['type'] === 'entity_reference') {
        $value = $share->get($hash_field['name'])->target_id;
      }
      else {
        $value = $share->get($hash_field['name'])->value;
      }
      $hash_components[] =  $value ? $value : 0;
    }
    return implode('-', $hash_components);
  }

  /**
   * {@inheritdoc}
   */
  public static function getShareHashComponents($share_hash) {
    $parts = explode('-', $share_hash);

    $fields = \Drupal::service('entity_field.manager')->getFieldDefinitions('share', 'share');

    $hash_components = [];
    foreach (static::getShareHashFields() as $k => $hash_field) {
      if (!in_array($hash_field['type'], ['entity_reference', 'boolean'], TRUE)) {
        continue;
      }
      if ($hash_field['type'] === 'entity_reference') {
        $storage = \Drupal::entityTypeManager()->getStorage(
          $fields[$hash_field['name']]->getSettings()['target_type']);
        $value = $storage->load($parts[$k]);
        $hash_components[] = $hash_field + [
          'value' => $parts[$k],
          'text' => $value ? $value->label() : 'None',
        ];
      }
      else {
        $hash_components[] = $hash_field + [
          'value' => $parts[$k],
          'text' => $parts[$k],
        ];
      }
    }

    return $hash_components;
  }

  /**
   * {@inheritdoc}
   */
  public static function getShareHashTextComponents($share_hash) {
    $components = static::getShareHashComponents($share_hash);
    return array_map(function ($comp) {
      return $comp['text'];
    },
      $components
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function getShareHashText($share_hash) {
    return implode(' / ', static::getShareHashTextComponents($share_hash));
  }


  /**
   * {@inheritdoc}
   */
  public static function groupShareIdsByHash($share_ids) {
    $grouped = [];
    foreach ($share_ids as $share_id) {
      $share = Share::load($share_id);
      $hash = static::getShareHash($share);
      if (!isset($grouped[$hash])) {
        $grouped[$hash] = [
          'hash' => $hash,
          'components' => static::getShareHashComponents($hash),
          'hash_text' => static::getShareHashText($hash),
          'share_ids' => [],
        ];
      }
      $grouped[$hash]['share_ids'][] = $share->id();
    }

    foreach ($grouped as $hash => &$group) {
      $group['share_numbers'] = \Drupal::service(
        'shareholder_register.formatter')->shareIdsToRanges(
          $grouped[$hash]['share_ids']);
      $group['count'] = count($group['share_ids']);
    }
    return $grouped;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSharesHashSummary($share_ids) {
    $grouped = static::groupShareIdsByHash($share_ids);

    $summary = [];
    foreach ($grouped as $group) {
      $summary[] = t(
        '@number shares @hash_text (@share_numbers)',[
          '@number' => count($group['share_ids']),
          '@hash' => $group['hash_text'],
          '@share_numbers' => $group['share_numbers'],
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function convertShares($shares, $conversion_fields, $date, $message) {
    $updated_shares = [];
    foreach ($shares as $share) {
      $should_save = FALSE;

      $fields = \Drupal::service('entity_field.manager')->getFieldDefinitions('share', 'share');
      foreach (static::getShareHashFields() as $hash_field) {
        if (!in_array($hash_field['type'], ['entity_reference', 'boolean'], TRUE)) {
          continue;
        }
        if (!isset($conversion_fields[$hash_field['name']])) {
          continue;
        }

        if ($hash_field['type'] === 'entity_reference') {
          if ($conversion_fields[$hash_field['name']] != $share->get($hash_field['name'])->target_id) {
            $share->set($hash_field['name'], $conversion_fields[$hash_field['name']]);
            $should_save = TRUE;
          }
        }
        else {
          if ($conversion_fields[$hash_field['name']] != $share->get($hash_field['name'])->value) {
            $share->set($hash_field['name'], $conversion_fields[$hash_field['name']]);
            $should_save = TRUE;
          }
        }
      }

      if ($should_save) {
        $share->set('revision_date', $date);
        $share->set('revision_log_message', $message);
        $share->setNewRevision(TRUE);
        $share->save();
        $updated_shares[] = $share;
      }
    }

    return $updated_shares;
  }
}
