<?php

namespace Drupal\Tests\shareholder_register\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\shareholder_register\Traits\LoadRegisterTrait;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;

use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register\Entity\ShareTransaction;
use Drupal\shareholder_register\Entity\ShareTransactionGroup;
use Drupal\shareholder_register\Exception\ShareholderRegisterInvalidSharesException;

/**
 * Tests share.
 *
 * @group shareholder_register
 */
class ShareTest extends KernelTestBase {

  use LoadRegisterTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  public static $modules = [
    'system',
    'field',
    'user',
    'file',
    'text',
    'datetime',
    'simple_workflows',
    'shareholder_register',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp() {
    parent::setUp();
    $this->traitSetUp();

    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_test_boolean',
      'entity_type' => 'share_transaction',
      'type' => 'boolean',
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_name' => 'field_test_boolean',
      'entity_type' => 'share_transaction',
      'bundle' => 'share_transaction',
      'label' => 'Test field',
    ]);
    $field->save();

    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_test_boolean',
      'entity_type' => 'share',
      'type' => 'boolean',
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_name' => 'field_test_boolean',
      'entity_type' => 'share',
      'bundle' => 'share',
      'label' => 'Test field',
    ]);
    $field->save();
  }

  /**
   * Setup test register for negative transactions.
   */
  protected function loadTestRegister() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
          [
            'number' => 2,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 2,
            'shareholder_number' => 1,
            'payment_date' => '2017-01-02',
            'validation_date' => '2017-01-02',
            'field_test_boolean' => 1,
          ],
          [
            'quantity' => 2,
            'shareholder_number' => 1,
            'payment_date' => '2017-02-02',
            'validation_date' => '2017-02-02',
            'field_test_boolean' => 0,
          ],
          // add an unvalidated repurchase with shares attached
          [
            'quantity' => -2,
            'shareholder_number' => 2,
          ],
        ],
      ]);
    return $register;
  }

  /**
   * Test copy of fields from transaction to share.
   */
  public function testCopyFieldsFromTransaction() {
    $register = $this->loadTestRegister();

    $t1 = reset($register['transactions']);
    $t2 = next($register['transactions']);

    foreach ($t1->getShares() as $s) {
      $this->assertEqual($s->get('field_test_boolean')->value, 1);
    }
    foreach ($t2->getShares() as $s) {
      $this->assertEqual($s->get('field_test_boolean')->value, 0);
    }
  }

  /**
   * Test current holder field of share.
   */
  public function testCurrentHolder() {
    $register = $this->loadTestRegister();

    $t = reset($register['transactions']);
    foreach ($t->getShares() as $s) {
      $this->assertEqual(
        $s->get('shareholder_id')->target_id,
        reset($register['shareholder'])
      );
    }

    $group = ShareTransactionGroup::create([
      'type' => 'transfer',
    ]);
    $group->save();

    $t1 = ShareTransaction::create([
           'quantity' => -2,
           'shareholder_id' => $register['shareholder']['1'],
           'payment_date' => '2018-01-01',
           'validation_date' => '2018-01-02',
           'transaction_group' => $group->id(),
    ]);
    $t1->save();
    $t2 = ShareTransaction::create([
           'quantity' => 2,
           'shareholder_id' => $register['shareholder']['2'],
           'payment_date' => '2018-01-01',
           'validation_date' => '2018-01-02',
           'transaction_group' => $group->id(),
    ]);
    $t2->save();

    $t1->actionValidate('2018-01-02');

    foreach ($t->getShares() as $s) {
      $this->assertEqual(
        $s->get('shareholder_id')->target_id,
        end($register['shareholder'])
      );
    }

    $repurchase = end($register['transactions']);
    $repurchase->attachSharesToTransaction('2019-01-01');

    foreach ($t->getShares() as $s) {
      $this->assertEqual(
        $s->get('shareholder_id')->target_id,
        end($register['shareholder'])
      );
    }

    $repurchase->actionValidate('2019-01-01');
    foreach ($t->getShares() as $s) {
      $this->assertEqual(
        $s->get('shareholder_id')->target_id,
        NULL
      );
    }
  }


}
