<?php

namespace Drupal\Tests\shareholder_register\Kernel;

use Drupal\shareholder_register\Entity\Shareholder;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\shareholder_register\Traits\LoadRegisterTrait;

use Drupal\shareholder_register\Exception\ShareholderRegisterInvalidSharesException;

/**
 * Tests share transaction.
 *
 * @group shareholder_register
 */
class ShareTransactionTest extends KernelTestBase {

  use LoadRegisterTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  public static $modules = [
    'system',
    'user',
    'file',
    'text',
    'datetime',
    'simple_workflows',
    'shareholder_register',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp() {
    parent::setUp();
    $this->traitSetUp();
  }

  /**
   * Setup test register for negative transactions.
   */
  protected function loadTestRegister() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2017-01-02',
            'validation_date' => '2017-02-02',
          ],
          [
            'quantity' => -2,
            'shareholder_number' => 1,
          ],
          [
            'quantity' => -2,
            'shareholder_number' => 1,
          ],
        ],
      ]);
    return $register;
  }

  /**
   * Test the Share Transaction dates.
   */
  public function testShareTransactionDates() {
    $register = $this->loadTestRegister();

    $t = reset($register['transactions']);

    $this->assertEqual($t->getDate(), '2017-02-02');
    $this->assertEqual($t->getPaymentDate(), '2017-01-02');
    $this->assertEqual($t->getTransactionDate(), '2017-02-02');
    $this->assertEqual($t->getTransactionDate(FALSE), '2017-02-02');
    $this->assertEqual($t->getTransactionDate(TRUE), '2017-01-02');

    $s = Shareholder::load(reset($register['shareholder']));
    $this->assertEqual(count($s->getValidTransactionsInRange('2017-01-01', '2017-01-31', FALSE)), 0);
    $this->assertEqual(count($s->getValidTransactionsInRange('2017-01-01', '2017-01-31', TRUE)), 1);

    $this->assertEqual(count($s->getValidTransactionsInRange('2017-02-01', '2017-02-28', FALSE)), 1);
    $this->assertEqual(count($s->getValidTransactionsInRange('2017-02-01', '2017-02-28', TRUE)), 0);
  }

  /**
   * Test validate Transaction.
   */
  public function testShareTransactionValidate() {
    $register = $this->loadTestRegister();

    $t1 = end($register['transactions']);
    $t2 = prev($register['transactions']);

    $this->assertEqual(count($t1->attachSharesToTransaction('2018-01-01')), abs($t1->getQuantity()));
    $t1->actionValidate('2018-01-01');

    $this->assertEqual(count($t2->attachSharesToTransaction('2018-01-01')), abs($t2->getQuantity()));
    $t2->actionValidate('2018-02-02');
  }

  /**
   * Test invalid shares attached to Transaction.
   */
  public function testInvalidShareTransactionValidate() {
    $register = $this->loadTestRegister();

    $t1 = end($register['transactions']);
    $t2 = prev($register['transactions']);

    $this->assertEqual(count($t1->attachSharesToTransaction('2018-01-01')), abs($t1->getQuantity()));
    $this->assertEqual(count($t2->attachSharesToTransaction('2018-01-01')), abs($t2->getQuantity()));

    $t1->actionValidate('2018-01-01');

    $this->expectException(ShareholderRegisterInvalidSharesException::class);
    $t2->actionValidate('2018-02-02');
  }

}
