<?php

namespace Drupal\Tests\shareholder_register\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\shareholder_register\Traits\LoadRegisterTrait;

use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register\Exception\ShareholderRegisterInvalidShareholderIdentifierException;

/**
 * Tests taxshelter claim creation.
 *
 * @group shareholder_register
 */
class ShareholderIdentifierTest extends KernelTestBase {

  use LoadRegisterTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  public static $modules = [
    'system',
    'user',
    'file',
    'text',
    'datetime',
    'simple_workflows',
    'shareholder_register'
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp() {
    parent::setUp();
    $this->traitSetUp();
  }

  /**
   * Setup test register.
   */
  protected function loadTestRegister() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
            'address' => [
              'given_name' => 'test',
              'family_name' => 'test',
              'postal_code' => '1000',
              'locality' => 'test',
              'country_code' => 'BE',
            ],
          ],
          [
            'number' => 2,
            'address' => [
              'given_name' => 'test',
              'family_name' => 'test',
              'postal_code' => '1000',
              'locality' => 'test',
              'country_code' => 'BE',
            ],
          ],
        ],
        'transactions' => [
          [
            'quantity' => 1,
            'shareholder_number' => 1,
            'payment_date' => '2019-02-02',
            'validation_date' => '2019-02-02',
          ],
          [
            'quantity' => 1,
            'shareholder_number' => 2,
            'payment_date' => '2020-02-02',
          ],
        ],
      ]);
    return $register;
  }

  /**
   * Test exception on validation of shareholder with duplicate identifier.
   */
  public function testDuplicateIdentifier() {
    $register = $this->loadTestRegister();
    $t = end($register['transactions']);

    $this->expectException(ShareholderRegisterInvalidShareholderIdentifierException::class);
    $t->actionValidate('2020-01-01');
  }

  /**
   * Test validation of shareholder with manual identifier.
   */
  public function testManualIdentifier() {
    $register = $this->loadTestRegister();
    $t = end($register['transactions']);

    // Manually set identifier.
    $t->getShareholder()->set('identifier', 'test2')->save();

    $t->actionValidate('2020-01-01');

    $shareholder = Shareholder::load($t->get('shareholder_id')->target_id);
    $this->assertEqual($shareholder->getState(), "valid");
    $this->assertEqual($shareholder->getIdentifier(), "test2");
  }

  /**
   * Test regeneration of identifier.
   */
  public function testRecreateIdentifier() {
    $register = $this->loadTestRegister();
    $t = end($register['transactions']);

    $shareholder = Shareholder::load($t->get('shareholder_id')->target_id);
    $this->assertEqual($shareholder->getIdentifier(), "test test", "Initial identifier = 'test test 2'");

    // Manually set identifier.
    $t->getShareholder()->set('identifier', 'test2')->save();
    $shareholder = Shareholder::load($t->get('shareholder_id')->target_id);
    $this->assertEqual($shareholder->getIdentifier(), "test2", "Manual identifier = 'test2'");

    // Recreate identifier.
    $t->getShareholder()->set('identifier', '')->save();
    $shareholder = Shareholder::load($t->get('shareholder_id')->target_id);
    $this->assertEqual($shareholder->getIdentifier(), "test test", "Recreated identifier = 'test test 2'");
  }

}
