<?php
namespace Drupal\banking_import_coda\Plugin\BankingImportFormat;

use Drupal\banking_import\Exception\BankingImportException;
use Drupal\banking_import\Plugin\BankingImportFormatDefinitionBase;

use Codelicious\Coda\Parser;

use Money\Money;
use Money\Currencies\ISOCurrencies;
use Money\Formatter\DecimalMoneyFormatter;

use Drupal\banking\Entity\BankTransaction;
use Drupal\banking\Entity\BankAccount;
use Drupal\banking\Event\BankTransactionEvent;

/**
 * Belgian CODA file import.
 *
 * @BankingImportFormatDefinition(
 *   id = "coda",
 *   label = @Translation("Coda"),
 *  )
 */
class Coda extends BankingImportFormatDefinitionBase {

  /**
   * {@inheritdoc}
   */
  public function validateFile($coda_file) {
    $parser = new Parser();
    $statements = $parser->parseFile($coda_file);

    $accounts = [];
    foreach ($statements as $statement) {
      $account = trim($statement->getAccount()->getNumber());
      if (!in_array($account, $accounts)) {
        $accounts[] = $account;
      }
    }

    foreach ($accounts as $account) {
      if (!BankAccount::loadByAccountNumber($account)) {
        throw new BankingImportException(t(
            "This Coda file contains statements for unknown bank account '@account'",
            [
              '@account' => $account,
            ]
        ));
      }
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function importFileBatch($file) {
    $coda_file = \Drupal::service('file_system')->realpath($file->GetFileUri());
    $parser = new Parser();
    $statements = $parser->parseFile($coda_file);

    $batch = array(
      'title' => t('Importing Coda Statement'),
      'operations' => [],
    );

    foreach ($statements as $statement) {

      $bank_account = BankAccount::loadByAccountNumber(trim($statement->getAccount()->getNumber()));
      $statement_from_date = $statement->getDate()->format('Y-m-d');
      $statement_to_date = $statement->getDate()->format('Y-m-d');
      $query = \Drupal::entityQuery('bank_transaction')
        ->accessCheck(FALSE)
        ->condition('date', $statement_from_date, '>=')
        ->condition('date', $statement_to_date, '<=')
        ->condition('bank_account_id', $bank_account->id());
      $values = $query->execute();
      $dates = [];
      foreach (BankTransaction::loadMultiple(array_keys($values)) as $transactie) {
        if (!in_array($transactie->getDate(), $dates)) {
          $dates[] = $transactie->getDate();
        }
      }
      $batch['operations'][] = [
        '\Drupal\banking_import_coda\Plugin\BankingImportFormat\Coda::importStatement',
        [
          $statement,
          $dates,
          $file->id(),
        ]
      ];
    }

    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public static function importStatement($statement, $dates, $file_id, &$context) {
    $n = 0;
    foreach ($statement->getTransactions() as $entry) {
      if (in_array($entry->getTransactionDate()->format('Y-m-d'), $dates)) {
        \Drupal::messenger()->addWarning(t("Not importing entry for day @day and account @account: there are already transactions for this day!", [
          '@day' => $entry->getTransactionDate()->format('Y-m-d'),
          '@account' => $statement->getAccount()->getNumber(),
        ]));
        continue;
      }

      $bank_account = BankAccount::loadByAccountNumber(trim($statement->getAccount()->getNumber()));
      $currencies = new ISOCurrencies();
      $moneyFormatter = new DecimalMoneyFormatter($currencies);
      $amount = Money::EUR(intval(round($entry->getAmount() * 100)));

      $transaction_data = [
        'name' => $statement->getAccount()->getNumber() . '/' . $statement->getDate()->format('Y-m-d') . '/' . $n,
        'bank_account_id' => $bank_account,
        'amount' => $moneyFormatter->format($amount),
        'date' => $entry->getTransactionDate()->format('Y-m-d'),
        'account_holder_name' => $entry->getAccount()->getName(),
        'account_number' => $entry->getAccount()->getNumber(),
        'information' => substr($entry->getMessage() . ' ' . $entry->getStructuredMessage(), 0, 4096),
        'import_file' => $file_id,
      ];

      \Drupal::service('banking_import.default')->createImportedTransaction($transaction_data);

      if (!isset($context['results']['count'])) {
        $context['results']['count'] = 0;
      }
      $context['results']['count'] += 1;

      $n++;
    }
  }

}
