<?php

namespace Drupal\banking_orders\Entity;

use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;

use Drupal\banking\Exception\BankingIntegrityException;

/**
 * Defines the Bank order entity.
 *
 * @ingroup banking_orders
 *
 * @ContentEntityType(
 *   id = "bank_order",
 *   label = @Translation("Bank order"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\banking_orders\BankOrderListBuilder",
 *     "views_data" = "Drupal\banking_orders\Entity\BankOrderViewsData",
 *
 *     "form" = {
 *       "default" = "Drupal\banking_orders\Form\BankOrderForm",
 *       "add" = "Drupal\banking_orders\Form\BankOrderForm",
 *       "edit" = "Drupal\banking_orders\Form\BankOrderForm",
 *       "delete" = "Drupal\banking_orders\Form\BankOrderDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\banking_orders\BankOrderHtmlRouteProvider",
 *     },
 *     "access" = "Drupal\banking_orders\BankOrderAccessControlHandler",
 *   },
 *   base_table = "bank_order",
 *   translatable = FALSE,
 *   admin_permission = "administer bank order entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *   },
 *   links = {
 *     "canonical" = "/admin/banking/tools/bank_order/{bank_order}",
 *     "add-form" = "/admin/banking/tools/bank_order/add",
 *     "edit-form" = "/admin/banking/tools/bank_order/{bank_order}/edit",
 *     "delete-form" = "/admin/banking/tools/bank_order/{bank_order}/delete",
 *     "collection" = "/admin/banking/tools/bank_order",
 *   },
 *   field_ui_base_route = "bank_order.settings"
 * )
 */
class BankOrder extends ContentEntityBase implements BankOrderInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public function delete() {
    if ($this->getState() != 'draft') {
      throw new BankingIntegrityException(
        "You cannot delete a non-draft Banking Order");
    }
    else {
      foreach ($this->get('lines')->referencedEntities() as $line) {
        $line->delete();
      }
      return parent::delete();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getBankAccount() {
    return $this->get('bank_account')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getAmount() {
    $value = 0;
    if (!$this->isNew()) {
      $connection = \Drupal::database();
      $r = $connection->query(
        "SELECT COALESCE(SUM(amount), 0) AS amount FROM {bank_order_line} WHERE bank_order_id = :order_id",
        [
          ':order_id' => $this->id(),
        ]);
      $computed_values = $r->fetchAssoc();
      $value = $computed_values['amount'];
    }
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  public function getLines() {
    return BankOrderLine::loadMultiple(
      $this->getLineIds());
  }

  /**
   * {@inheritdoc}
   */
  public function getLineIds() {
    return \Drupal::entityQuery('bank_order_line')
      ->accessCheck(FALSE)
      ->condition('bank_order_id', $this->id())
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function getState() {
    return $this->get('state')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setState($state) {
    $this->set('state', $state);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the Bank order entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['order_type'] = BaseFieldDefinition::create('string')
      ->setLabel(t('The Order Type'))
      ->setDescription(t('The type of the Bank order (debit / credit)).'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('credit')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['bank_account'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Bank Account'))
      ->setDescription(t('The Bank Account.'))
      ->setSettings([
        'target_type' => 'bank_account',
        'default_value' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['lines'] = BaseFieldDefinition::create('bank_order_line')
      ->setLabel(t('Lines'))
      ->setComputed(TRUE)
      ->setSettings([
        'target_type' => 'bank_order_line',
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'bank_order_line_formatter',
        'label' => 'above',
        'weight' => -5,
      ]);

    $fields['export'] = BaseFieldDefinition::create('file')
      ->setLabel(t('The exported Order file.'))
      ->setDescription(t('The Order.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['state'] = BaseFieldDefinition::create('simple_workflow_state_field')
      ->setLabel(t('State'))
      ->setDescription(t('State of the Bank Order.'))
      ->setDefaultValue('draft');

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

}
