<?php

namespace Drupal\banking_orders\Entity;

use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;

use Drupal\banking\Exception\BankingIntegrityException;

/**
 * Defines the Bank order line entity.
 *
 * @ingroup banking_orders
 *
 * @ContentEntityType(
 *   id = "bank_order_line",
 *   label = @Translation("Bank order line"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\banking_orders\BankOrderLineListBuilder",
 *     "views_data" = "Drupal\banking_orders\Entity\BankOrderLineViewsData",
 *
 *     "form" = {
 *       "default" = "Drupal\banking_orders\Form\BankOrderLineForm",
 *       "add" = "Drupal\banking_orders\Form\BankOrderLineForm",
 *       "edit" = "Drupal\banking_orders\Form\BankOrderLineForm",
 *       "delete" = "Drupal\banking_orders\Form\BankOrderLineDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\banking_orders\BankOrderLineHtmlRouteProvider",
 *     },
 *     "access" = "Drupal\banking_orders\BankOrderLineAccessControlHandler",
 *   },
 *   base_table = "bank_order_line",
 *   translatable = FALSE,
 *   admin_permission = "administer bank order line entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *   },
 *   links = {
 *     "canonical" = "/admin/banking/tools/bank_order_line/{bank_order_line}",
 *     "add-form" = "/admin/banking/tools/bank_order_line/add",
 *     "edit-form" = "/admin/banking/tools/bank_order_line/{bank_order_line}/edit",
 *     "delete-form" = "/admin/banking/tools/bank_order_line/{bank_order_line}/delete",
 *     "collection" = "/admin/banking/tools/bank_order_line",
 *   },
 *   field_ui_base_route = "bank_order_line.settings"
 * )
 */
class BankOrderLine extends ContentEntityBase implements BankOrderLineInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public function delete() {
    if ($this->getOrder() && $this->getOrder()->getState() != 'draft') {
      throw new BankingIntegrityException(
        "You cannot delete a line from a non-draft Banking Order Line");
    }
    else {
      return parent::delete();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountHolder() {
    return $this->get('accountholder')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountNumber() {
    return $this->get('iban')->iban;
  }

  /**
   * {@inheritdoc}
   */
  public function getIbanBic() {
    return $this->get('iban')->first()->getValue();
  }

  /**
   * {@inheritdoc}
   */
  public function getAmount() {
    return $this->get('amount')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getOrder() {
    return $this->get('bank_order_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getState() {
    if ($this->getOrder()) {
      return $this->getOrder()->getState();
    }
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function getData($key, $default = NULL) {
    $data = [];
    if (!$this->get('extra')->isEmpty()) {
      $data = $this->get('extra')->first()->getValue();
    }
    return $data[$key] ?? $default;
  }

  /**
   * {@inheritdoc}
   */
  public function setData($key, $value) {
    $this->get('extra')->__set($key, $value);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the Bank order line entity.'))
      ->setSettings([
        'max_length' => 140,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Execution Date'))
      ->setDescription(t('The requested execution date.'))
      ->setSettings([
        'datetime_type' => 'date',
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'datetime_default',
        'settings' => [
          'format_type' => 'date',
        ],
        'weight' => 4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['endtoendid'] = BaseFieldDefinition::create('string')
      ->setLabel(t('End to End ID'))
      ->setDescription(t('The End to End ID of the Bank order line entity.'))
      ->setSettings([
        'max_length' => 35,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['bank_order_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Bank Order'))
      ->setDescription(t('The Bank Order.'))
      ->setSettings([
        'target_type' => 'bank_order',
        'default_value' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 2,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['accountholder'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Accountholder Name'))
      ->setDescription(t('The name of the Accountholder.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['iban'] = BaseFieldDefinition::create('iban_bic')
      ->setLabel(t('IBAN'))
      ->setDescription(t('The Account Number of the other Party.'))
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'iban_bic_widget',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['amount'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Amount'))
      ->setDescription(t('The Amount of the transaction.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['extra'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Extra'))
      ->setDescription(t('Extra data for this Bank Order Line.'));

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

}
