<?php

namespace Drupal\banking_orders\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormState;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;

use Drupal\Component\Utility\NestedArray;

use Drupal\banking_orders\Entity\BankOrder;

/**
 * Class ExportBankOrderForm.
 */
class ExportBankOrderForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'export_bank_order_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, BankOrder $bank_order = NULL) {
    $form['bank_order'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Bank Order'),
      '#weight' => '0',
      '#target_type' => 'bank_order',
      '#required' => TRUE,
      '#default_value' => $bank_order,
    ];

    $this->manager = \Drupal::service('plugin.manager.bank_order_export_plugin');

    if ($bank_order) {
      $export_options = array_column($this->manager->getApplicableDefinitions(
                          $bank_order->get('order_type')->value), 'label', 'id');
    }
    else {
      $export_options = array_column($this->manager->getDefinitions(),
                        'label', 'id');
    }

    $wrapper_id = 'ajax-wrapper';
    $form['plugin'] = [
      '#type' => 'select',
      '#title' => $this->t('Plugin'),
      '#options' => $export_options,
      '#weight' => '15',
      '#required' => TRUE,
      '#ajax' => [
        'callback' => [get_called_class(), 'updatePlugin'],
        'wrapper' => $wrapper_id,
      ],
    ];
    $form['plugin_settings_wrapper'] = [
      '#weight' => '20',
      '#tree' => TRUE,
      '#prefix' => '<div id="' . $wrapper_id . '">',
      '#suffix' => '</div>',
    ];

    if ($form_state->getValue('plugin')) {
      $plugin_id = $form_state->getValue('plugin');
      $plugin_values = $form_state->getValue(['plugin_settings_wrapper']);
      $plugin = $this->manager->createInstance($plugin_id, $plugin_values ? $plugin_values : []);
      if ($plugin instanceof PluginFormInterface) {
        $form['plugin_settings_wrapper'] += $plugin->buildConfigurationForm([], $form_state);
      }
    }

    $form['submit'] = [
      '#weight' => '100',
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public static function updatePlugin(array $form, FormStateInterface $form_state) {
    $subform = NestedArray::getValue($form, ['plugin_settings_wrapper']);
    return $subform;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $order = BankOrder::load($form_state->getValue('bank_order'));
    if (!$order->get('bank_account')->entity) {
      $form_state->setErrorByName(
        'bank_order',
        $this->t("Bank order doesn't have bank account set!")
      );
    }

    $this->manager = \Drupal::service('plugin.manager.bank_order_export_plugin');
    if ($form_state->getValue('plugin')) {
      $plugin_id = $form_state->getValue('plugin');
      $plugin = $this->manager->createInstance($plugin_id, []);
      $plugin_values = $form_state->getValue(['plugin_settings_wrapper']);
      if ($plugin_values) {
        $plugin->validateConfigurationForm($form, $form_state);
      }
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->manager = \Drupal::service('plugin.manager.bank_order_export_plugin');
    if ($form_state->getValue('plugin')) {
      $plugin_id = $form_state->getValue('plugin');
      $plugin = $this->manager->createInstance($plugin_id, []);
      $plugin_values = $form_state->getValue(['plugin_settings_wrapper']);
      if ($plugin_values) {
        $sub_form_state = new FormState();
        $sub_form_state->setValues($plugin_values);
        $plugin->submitConfigurationForm($form, $sub_form_state);
      }
    }
    $order = BankOrder::load($form_state->getValue('bank_order'));
    $form_state->setRedirect('entity.bank_order.canonical', ['bank_order' => $order->id()]);

    $batch = $plugin->getBankOrderExportBatch($order);
    batch_set($batch);
  }

}
