<?php

namespace Drupal\banking_orders\Plugin\BankOrderExportPlugin;

use Drupal\Core\Link;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

use AbcAeffchen\Sephpa\SephpaCreditTransfer;
use AbcAeffchen\SepaUtilities\SepaUtilities;

use Drupal\banking_orders\Plugin\BankOrderExportPluginInterface;

use Drupal\banking_orders\Entity\BankOrder;
use Drupal\banking_orders\Entity\BankOrderLine;

/**
 * Export SEPA SCT.
 *
 * @BankOrderExportPlugin(
 *   id = "sepa_sct",
 *   label = @Translation("SEPA SCT"),
 *   types = {
 *     "credit",
 *   },
 *  )
 */
class SepaSct extends PluginBase implements PluginFormInterface, BankOrderExportPluginInterface {
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['version'] = [
      '#type' => 'select',
      '#title' => $this->t('Select version'),
      '#required' => TRUE,
      '#options' => [
        SephpaCreditTransfer::SEPA_PAIN_001_001_03 => $this->t('SEPA_PAIN_001_001_03'),
        SephpaCreditTransfer::SEPA_PAIN_001_002_03 => $this->t('SEPA_PAIN_001_002_03'),
        SephpaCreditTransfer::SEPA_PAIN_001_003_03 => $this->t('SEPA_PAIN_001_003_03'),
      ],
      '#default_value' => SephpaCreditTransfer::SEPA_PAIN_001_001_03,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['version'] = $form_state->getValue('version');
  }

  /**
   * {@inheritdoc}
   */
  public function getBankOrderExportBatch(BankOrder $order) {
    $batch = [
      'title' => $this->t('Exporting Bank Order'),
      'operations' => [],
    ];

    $batch['operations'][] = [
      '\Drupal\banking_orders\Plugin\BankOrderExportPlugin\SepaSct::addBankOrderLines',
      [
        $order->id(),
        $this->configuration['version'],
      ],
    ];
    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public static function addBankOrderLines($bank_order_id, $version, &$context) {
    $order = BankOrder::load($bank_order_id);

    if (!isset($context['sandbox']['lines'])) {
      $line_ids = \Drupal::entityQuery('bank_order_line')
        ->accessCheck(FALSE)
        ->condition('bank_order_id', $bank_order_id)
        ->execute();
      $context['sandbox']['lines'] = $line_ids;
      $context['sandbox']['total'] = count($context['sandbox']['lines']);

      $context['sandbox']['file'] = new SephpaCreditTransfer(
        'Initiator Name',
        "{$order->getName()}",
        $version
      );
      $context['sandbox']['collections'] = [];
    }

    for ($i = 0; $i < 100; $i++) {
      if ($line_id = array_pop($context['sandbox']['lines'])) {

        $line = BankOrderLine::load($line_id);

        if ($line->get('amount')->value < 0.01) {
          continue;
        }

        $iban = iban_to_machine_format($line->getIbanBic()['iban']);
        $bic = iban_to_machine_format($line->getIbanBic()['bic']);

        if (!SepaUtilities::checkIban($iban) ||
          !SepaUtilities::checkBIC($bic) ||
          !SepaUtilities::crossCheckIbanBic($iban, $bic)) {
          \Drupal::messenger()->addMessage(
            "Unknown or invalid account number '{$iban}' / '{$bic}' for line '{$line->getName()}'!",
            'error'
          );
          return;
        }

        // Create collection if it doesn't exit yet.
        $date = $line->get('date')->value ?: 'asap';
        if (!array_key_exists($date, $context['sandbox']['collections'])) {
          $collection_data = [
              // Required information about the debtor.
              // ID of the payment collection.
            'pmtInfId' => str_replace('-', ' ', "{$date}"),
              // Debtor (max 70 characters).
            'dbtr' => $order->get('bank_account')->entity->get('name')->value,
              // IBAN of the Debtor.
            'iban' => preg_replace('/\s+/', '', $order->get('bank_account')->entity->get('number')->value),
            'ccy' => 'EUR',
              // BatchBooking, only 'true' or 'false'.
            'btchBookg' => 'false',
          ];
          if ($date !== 'asap') {
            $collection_data['reqdExctnDt'] = $date;
          }
          $collection = $context['sandbox']['collections'][$date] = $context['sandbox']['file']->addCollection($collection_data);
        }
        $collection = $context['sandbox']['collections'][$date];

        $pmtId = $line->getData('pmtId');

        $collection->addPayment([
            // Required information about the creditor.
            // ID of the payment (EndToEndId).
          'pmtId' => $pmtId ? $pmtId : $line->id(),
          'instdAmt' => $line->get('amount')->value,
          'iban' => $iban,
          'bic' => $bic,
            // Creditor (max 70 characters)
          'cdtr' => $line->get('accountholder')->value,
            // Unstructured information about the remittance (max 140 characters).
          'rmtInf' => $line->getName(),
        ]);
      }
    }

    if ($context['sandbox']['total'] > 0) {
      $context['finished'] = ($context['sandbox']['total'] - count($context['sandbox']['lines'])) / $context['sandbox']['total'];
    }
    else {
      $context['finished'] = 1;
    }

    if ($context['finished'] == 1) {
      $options = [];
      try {
        $data = $context['sandbox']['file']->generateOutput($options);
      }
      catch (\Exception $e) {
        $this->messenger()->addMessage(
          $e->getMessage(),
          'error'
        );
      }

      $output = \Drupal::service('file.repository')->writeData(reset($data)['data'], 'private://payment_order.xml');

      $order->set('export', $output);
      $order->save();

      \Drupal::messenger()->addMessage(t(
          'Exported payment order @url',
          [
            '@url' => Link::fromTextAndUrl(
              'payment_order.xml',
              \Drupal::service('file_url_generator')->generate($output->getFileUri()))->toString(),
          ]
      ));
    }
  }

}
