<?php

namespace Drupal\banking_orders\Plugin\BankOrderExportPlugin;

use Drupal\Core\Link;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

use AbcAeffchen\Sephpa\SephpaDirectDebit;
use AbcAeffchen\SepaUtilities\SepaUtilities;

use Drupal\banking_orders\Plugin\BankOrderExportPluginInterface;

use Drupal\banking_orders\Entity\BankOrder;
use Drupal\banking_orders\Entity\BankOrderLine;

/**
 * Export SEPA SDD.
 *
 * @BankOrderExportPlugin(
 *   id = "sepa_sdd",
 *   label = @Translation("SEPA SDD"),
 *   types = {
 *     "debit",
 *   },
 *  )
 */
class SepaSdd extends PluginBase implements PluginFormInterface, BankOrderExportPluginInterface {
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {

    $form['version'] = [
      '#type' => 'select',
      '#title' => $this->t('Select version'),
      '#required' => TRUE,
      '#options' => [
        SephpaDirectDebit::SEPA_PAIN_008_001_02 => $this->t('SEPA_PAIN_008_001_02'),
        SephpaDirectDebit::SEPA_PAIN_008_002_02 => $this->t('SEPA_PAIN_008_002_02'),
        SephpaDirectDebit::SEPA_PAIN_008_003_02 => $this->t('SEPA_PAIN_008_003_02'),
      ],
      '#default_value' => SephpaDirectDebit::SEPA_PAIN_008_002_02,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['version'] = $form_state->getValue('version');
  }

  /**
   * {@inheritdoc}
   */
  public function getBankOrderExportBatch(BankOrder $order) {
    $batch = [
      'title' => $this->t('Exporting Bank Order'),
      'operations' => [],
    ];

    $batch['operations'][] = [
      '\Drupal\banking_orders\Plugin\BankOrderExportPlugin\SepaSdd::addBankOrderLines',
      [
        $order->id(),
        $this->configuration['version'],
      ],
    ];
    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public static function addBankOrderLines($bank_order_id, $version, &$context) {
    $order = BankOrder::load($bank_order_id);

    if (!isset($context['sandbox']['lines'])) {
      $line_ids = \Drupal::entityQuery('bank_order_line')
        ->condition('bank_order_id', $bank_order_id)
        ->execute();
      $context['sandbox']['lines'] = $line_ids;
      $context['sandbox']['total'] = count($context['sandbox']['lines']);

      $cred_iban = $order->get('bank_account')->entity->get('iban')->first()->getValue();
      $context['sandbox']['file'] = new SephpaDirectDebit(
        'Initiator Name',
        "{$order->getName()}",
        $version
      );
      $context['sandbox']['collection'] = $context['sandbox']['file']->addCollection(
        [
          'pmtInfId' => $order->getName(),
          'lclInstrm' => SepaUtilities::LOCAL_INSTRUMENT_CORE_DIRECT_DEBIT,
          'seqTp' => SepaUtilities::SEQUENCE_TYPE_RECURRING,
          'cdtr' => $order->get('bank_account')->entity->get('name')->value,
          'iban' => $cred_iban['iban'],
          'bic' => $cred_iban['bic'],
          'ci' => $order->get('bank_account')->entity->get('creditor_identification')->value,
        ]
      );
    }

    for ($i = 0; $i < 100; $i++) {
      if ($line_id = array_pop($context['sandbox']['lines'])) {

        $line = BankOrderLine::load($line_id);

        if ($line->get('amount')->value < 0.01) {
          continue;
        }

        $iban = iban_to_machine_format($line->getIbanBic()['iban']);
        $bic = iban_to_machine_format($line->getIbanBic()['bic']);

        if (!SepaUtilities::checkIban($iban) ||
          !SepaUtilities::checkBIC($bic) ||
          !SepaUtilities::crossCheckIbanBic($iban, $bic)) {
          $this->messenger()->addMessage(
            "Unknown or invalid account number '{$iban}' / '{$bic}' for line '{$line->getName()}'!",
            'error'
          );
          return;
        }

        $pmtId = $line->getData('pmtId');

        $context['sandbox']['collection']->addPayment(
          [
            'pmtId' => $pmtId ? $pmtId : $line->id(),
            'instdAmt' => $line->get('amount')->value,
            'mndtId' => $line->getData('mndtId', FALSE),
            'dtOfSgntr' => $line->getData('dtOfSgntr', FALSE),
            'dbtr' => $line->get('accountholder')->value,
            'iban' => $iban,
            'bic' => $bic,
            'rmtInf' => $line->getName(),
          ]
        );
      }
    }

    if ($context['sandbox']['total'] > 0) {
      $context['finished'] = ($context['sandbox']['total'] - count($context['sandbox']['lines'])) / $context['sandbox']['total'];
    }
    else {
      $context['finished'] = 1;
    }

    if ($context['finished'] == 1) {
      $options = [];
      $data = $context['sandbox']['file']->generateOutput($options);
      $output = \Drupal::service('file.repository')->writeData(reset($data)['data'], 'private://payment_order.xml');

      $order->set('export', $output);
      $order->save();

      \Drupal::messenger()->addMessage(t(
          'Exported payment order @url',
          [
            '@url' => Link::fromTextAndUrl(
              'payment_order.xml',
              \Drupal::service('file_url_generator')->generate($output->getFileUri()))->toString(),
          ]
      ));
    }
  }

}
